/**
 * \file: util_mpf.h
 * 
 * \version: $Revision: 1.9 $
 * 
 * \release: $Name: UTIL_r2_E05 $
 *
 * Interface for a fixed memory pool implementation.
 * 
 * \component: Utility
 * 
 * \author T. Polle / ADIT / SWG / tpolle(o)de.adit-jv.com
 * 
 * \copyright: (c) 2003 - 2008 ADIT Corporation 
 */

#ifndef UTIL_MPF_H
#define UTIL_MPF_H

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \ingroup utility
 * \defgroup util_mpf Fixed Memory Pool
 *
 * The module provides a fixed memory pool implementation. The memory for the
 * pool is allocated when the pool is created.
 *
 * \{
 */

/**
 * The memory is externally provided through field \a mem in UTIL_CMPF_t.
 */
#define UTIL_TA_BUF   0x80000000U

/**
 * Fixed size memory pool specification.
 */
typedef struct
{
  VP   exinf;      /**< Extended information */
  ATR  mpfatr;     /**< Memory pool attribute */
  S32  mpfcnt;     /**< Number of blocks in whole memory pool */
  S32  blfsz;      /**< Fixed size memory block size (byte) */
  U8   dsname[8];  /**< Object name */
  VP   mem;        /**< Memory provided for the pool. Valid if
                        UTIL_TA_BUF is specified. */
} UTIL_CMPF_t;

typedef struct UTIL_mpf_t UTIL_mpf_t;

/**
 * Create a fixed size memory pool.
 * 
 * Creates a fixed size memory. \a pk_cmpf specifies the parameters of
 * the pool.  lock is not held.
 * 
 * \param pk_cmpf        pool parameters
 *
 * \returns \li (not NULL): a pointer to the memory pool
 *          \li NULL:       the pool could not be created
 */
IMPORT UTIL_mpf_t* UTIL_cre_mpf(UTIL_CMPF_t* pk_cmpf);
  
/**
 * (Re-)initialise a fixed size memory pool.
 * 
 * (Re-)initialise a fixed size memory. All allocated memory areas are
 * freed.
 * 
 * \param pool           pool
 */
IMPORT void UTIL_init_mpf(UTIL_mpf_t* pool);

/**
 * Delete \a pool.
 * 
 * \param pool           the pool
 *
 * \returns \li E_OK:                 \a pool deleted.
 *          \li T-Kernel error codes: error
 */
IMPORT S32 UTIL_del_mpf(UTIL_mpf_t* pool);
  
/**
 * Allocate a block from \a pool.
 * 
 * \param pool           the pool
 * \param block          alocated memory block
 * \param tmout          time out value (ignored)
 *
 * \returns \li E_OK:    \a block allocated.
 *          \li E_NOMEM: \a pool's memory exhausted
 */
IMPORT S32 UTIL_get_mpf(UTIL_mpf_t* pool, VP* block, TMO tmout);

/**
 * Free \a block from \a pool.
 * 
 * \param pool           the pool
 * \param block          alocated memory block
 *
 * \returns E_OK
 */
IMPORT S32 UTIL_rel_mpf(UTIL_mpf_t* pool, VP block_);

/**
 * Retrieves the size of internal overhead needed for a memory block.
 * 
 * \returns internal overhead
 */
IMPORT U32 UTIL_sizeof_cmpf(void);

/**
 * Get the block size of \a pool.
 * 
 * \param pool           the pool
 *
 * \returns block size
 */
IMPORT S32 UTIL_blfsz_mpf(UTIL_mpf_t* pool);

/**
 * \}
 */

#ifdef __cplusplus
};
#endif

#endif /* UTIL_MPF_H */
